//
//  ALYToastView.swift
//  TongyiApp
//
//  Created by ALY-Admin on 2023/9/13.
//  Copyright © 2023 Aliyun. All rights reserved.
//

import UIKit
import SnapKit

let K_Width = UIScreen.main.bounds.size.width
let K_Height = UIScreen.main.bounds.size.height

class ALYToast: NSObject {
    
    /// Toast类型
    enum Style: String {
        /// 成功
        case success
        /// 警告
        case warning
        /// 信息
        case info
        /// 错误
        case error
    }
    
    /// 亮暗(深浅)类型
    enum UIStyle: String {
        /// 亮(浅)色
        case light
        /// 暗(深)色
        case dark
    }
    
    static var shared = ALYToast()
    private var toastView: ALYToastView?
    
    /// - Parameters: 显示Toast
    ///   - text:  Toast文本信息
    ///   - style: 设置样式, 默认是信息图标
    ///   - uiStyle: 图标亮暗模式,默认是亮色
    func showToast(text: String,
                   style: ALYToast.Style = .info,
                   uiStyle: ALYToast.UIStyle = .light) {
        DispatchQueue.main.async {
            self.hide()
            self.createToastView(text: text, style: style, uiStyle: uiStyle)
        }
    }
    
    /// 创建toastView
    private func createToastView(text: String,
                                 style: ALYToast.Style,
                                 uiStyle: ALYToast.UIStyle) {
        self.toastView = ALYToastView()
        self.toastView?.titleLabel.text = text
        let imageNmame = "toast_\(style.rawValue)_\(uiStyle.rawValue)"
        self.toastView?.iconView.image = UIImage(named: imageNmame)
        var parentView: UIView?
        if let window = UIWindow.aly_currentWindow {
            parentView = window
        } else if let window = UIWindow.aly_appWindow {
            parentView = window
        } else if let topVC = aly_topmostViewController() {
            parentView = topVC.view
        }
        
        if let view = parentView {
            view.addSubview(self.toastView ?? ALYToastView())
            self.toastView?.snp.makeConstraints { make in
                make.center.equalToSuperview()
            }
        }
        
        var duration: TimeInterval = 3.0
        if text.count < 12 {
            duration = 2.0
        } else {
            duration = 3.0
        }
        NSObject.cancelPreviousPerformRequests(withTarget: self)
        self.perform(#selector(self.hide), with: nil, afterDelay: duration)
    }
    
    /// 隐藏
    @objc func hide() {
        self.toastView?.isHidden = true
        self.toastView?.removeFromSuperview()
        self.toastView = nil
    }
    
}


class ALYToastView: UIView {
    
    private lazy var backView: UIView = {
        let view = UIView()
        view.backgroundColor = .clear
        self.addSubview(view)
        return view
    }()
    
    lazy var iconView: UIImageView = {
        let iv = UIImageView()
        iv.image = UIImage(named: "toast_info_light")
        iv.contentMode = .scaleAspectFit
        return iv
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = .systemFont(ofSize: 14, weight: .regular)
        label.textColor = UIColor.aly_fromHex(0xFFFFFF)
        label.text = ""
        label.numberOfLines = 0
        label.textAlignment = .left
        return label
    }()
    
    /// 背景毛玻璃
    private lazy var visualEffectView: UIVisualEffectView = {
        let blur = UIBlurEffect(style: .systemMaterialDark)
        let blurView = UIVisualEffectView(effect: blur)
        blurView.contentView.backgroundColor = UIColor.aly_fromHex(0x2C2C36, alpha: 0.8).withAlphaComponent(0.2)
        blurView.layer.masksToBounds = true
        blurView.layer.cornerRadius = 12
        return blurView
    }()
    
    public override init(frame: CGRect) {
        super.init(frame: frame)
        
        self.backView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        self.backView.addSubview(self.visualEffectView)
        self.backView.addSubview(self.iconView)
        self.backView.addSubview(self.titleLabel)
        
        self.visualEffectView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        self.iconView.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.leading.equalToSuperview().inset(12)
            make.size.equalTo(CGSize(width: 20, height: 20))
        }
        self.titleLabel.snp.makeConstraints { make in
            make.top.bottom.equalToSuperview().inset(11)
            make.leading.equalTo(iconView.snp.trailing).offset(8)
            make.height.greaterThanOrEqualTo(18)
            make.width.lessThanOrEqualTo(230)
            make.trailing.equalToSuperview().inset(12)
        }
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
}
