//
//  String+ALYExt.swift
//  ALYFoundation
//
//  Created by Wien on 2023/9/7.
//  Copyright © 2023 Aliyun. All rights reserved.
//

import Foundation
import CryptoKit

// MARK: - Base64

extension String {
    
    /// base64转码
    public func aly_base64Encode() -> String {
        let result = Data(self.utf8).base64EncodedString()
        return result
    }
    
    /// base64解码
    public func aly_base64Decode() -> String? {
        guard let data = Data(base64Encoded: self) else {
            return nil
        }
        let result = String(data: data, encoding: .utf8)
        return result
    }
    
}


extension String {

    func aly_substring(from index: String.Index) -> String {
        let str = self[index...]
        return String(str)
    }

    func aly_substring(to index: String.Index) -> String {
        let str = self[..<index]
        return String(str)
    }

    func aly_substring(with aRange: Range<String.Index>) -> String {
        let str = self[aRange.lowerBound..<aRange.upperBound]
        return String(str)
    }
    
    /// 获取字符串中指定字串所在的ranges
    func aly_ranges(of substring: String, options: CompareOptions = [], locale: Locale? = nil) -> [Range<Index>] {
        var ranges: [Range<Index>] = []
        while ranges.last.map({ $0.upperBound < self.endIndex }) ?? true,
            let range = self.range(of: substring, options: options, range: (ranges.last?.upperBound ?? self.startIndex)..<self.endIndex, locale: locale)
        {
            ranges.append(range)
        }
        return ranges
    }
    
    static func aly_stringWith(_ data: [UInt8]) -> String {
        var str = ""
        for b in data {
            str.append(Character(UnicodeScalar(b)))
        }
        return str
    }
    
}


extension NSAttributedString {
    
    func aly_substring(to index: Int) -> NSAttributedString {
        var result = self
        let length: Int = index + 1
        if length <= self.length {
            result = self.attributedSubstring(from: NSRange(location: 0, length: length))
        }
        return result
    }
    
}


extension String {
    
    public func aly_trim() -> String {
        var resultString = self.trimmingCharacters(in: CharacterSet.whitespaces)
        resultString = resultString.trimmingCharacters(in: CharacterSet.newlines)
        return resultString
    }
    
    /// 判断是否包含另一个字符串
    /// - Parameters:
    ///   - str: 另一个字符串
    ///   - caseSensitive: 是否大小写敏感
    /// - Returns: true - 包含，false - 不包含
    public func aly_contains(_ str: String?, caseSensitive: Bool = true) -> Bool {
        guard let str = str else {
            return false
        }
        
        var str1 = self
        var str2 = str
        if !caseSensitive {
            str1 = self.lowercased()
            str2 = str.lowercased()
        }
        
        if str1.range(of: str2) != nil {
            return true
        }
        return false
    }
    
}

// MARK: - URL

extension String {
    
    public func aly_urlEncode() -> String? {
        let set = CharacterSet.alphanumerics.union(CharacterSet(charactersIn: "~-_."))
        return self.addingPercentEncoding(withAllowedCharacters: set)
    }
    
    public func aly_urlDecode() -> String? {
        return self.removingPercentEncoding
    }
    
    /// 为URL字符串增加末尾的查询参数
    /// - Parameters:
    ///   - name: 参数名称
    ///   - value: 参数值
    /// - Returns: URL字符串
    func aly_URLAddOrUpdateQueryStringParameter(name: String, value: String) -> String {
        guard var components = URLComponents(string: self) else {
            return self
        }
        
        let queryItem = URLQueryItem(name: name, value: value)
        
        if let items = components.queryItems {
            var tmpItems = items.filter { $0.name != name }
            tmpItems.append(queryItem)
            components.queryItems = tmpItems
        } else {
            components.queryItems = [queryItem]
        }
        
        return components.string ?? self
    }
    
    /// 为URL字符串增加末尾的查询参数
    /// - Parameter params: 参数名及参数值的Dictionary
    /// - Returns: URL字符串
    func aly_URLAddOrUpdateQueryStringParameter(params: [String: String]) -> String {
        guard var components = URLComponents(string: self) else {
            return self
        }
        
        let paramsItems = params.map { (key: String, value: String) in
            return URLQueryItem(name: key, value: value)
        }
        
        if let items = components.queryItems {
            var tmpItems = items.filter { params[$0.name] == nil }
            tmpItems.append(contentsOf: paramsItems)
            components.queryItems = tmpItems
        } else {
            components.queryItems = paramsItems
        }
        
        return components.string ?? self
    }
    
    /// 移除URL中的查询参数
    /// - Parameter params: 要移除的参数名列表
    /// - Returns: 处理后的结果。如果处理失败，则返回原字符串
    func aly_URLRemoveQueryStringParameter(names: [String]) -> String {
        guard var components = URLComponents(string: self) else {
            return self
        }
        
        var params = [String: String]()
        names.forEach { name in
            params[name] = "1"
        }
        
        if let items = components.queryItems {
            let tmpItems = items.filter { params[$0.name] == nil }
            components.queryItems = tmpItems
        }
        
        if let str = components.string {
            return str
        } else {
            return self
        }
    }
    
    /// 移除URL中的查询参数
    /// - Parameter name: 要移除的参数名
    /// - Returns: 处理后的结果。如果处理失败，则返回原字符串
    func aly_URLRemoveQueryStringParameter(name: String) -> String {
        return aly_URLRemoveQueryStringParameter(names: [name])
    }
    
}


extension String {
    
    public func aly_replaceWithRegexPattern(_ pattern: String, template: String) -> String {
        var finalStr = self
        do {
            let regex = try NSRegularExpression(pattern: pattern, options: NSRegularExpression.Options.caseInsensitive)
            finalStr = regex.stringByReplacingMatches(in: self,
                                                      options: NSRegularExpression.MatchingOptions(rawValue: 0),
                                                      range: NSMakeRange(0, self.utf16.count),
                                                      withTemplate: template)
        }
        catch {
            print(error)
        }
        return finalStr
    }
    
}


extension NSMutableAttributedString {
        
    @objc private func alyfix_attributes(at location: Int, effectiveRange range: NSRangePointer?) -> [NSAttributedString.Key : Any] {
        if self.length == 0 {
            return [:]
        }
        return self.alyfix_attributes(at: location, effectiveRange: range)
    }
    
}


extension String {
    
    // 转换为基本类型
    public func aly_toBool() -> Bool? {
        switch self {
        case "TRUE", "True", "true", "YES", "Yes", "yes", "1":
            return true
        case "FALSE", "False", "false", "NO", "No", "no", "0":
            return false
        default:
            return nil
        }
    }
    
    public func aly_toInt() -> Int? {
        return Int(self)
    }

    public func aly_toDouble() -> Double? {
        return Double(self)
    }

    public func aly_toFloat() -> Float? {
        return Float(self)
    }

    public func aly_toInt64() -> Int64? {
        return Int64(self)
    }
    
}

// MARK: - Digest

extension String {
    
    /// clean special characters from string
    public func aly_clean() -> String {
        let patterns: CharacterSet = ["$", "#", "@", "!", "&", "*", "(", ")", "^", "%", "/"]
        let separatedStrings = self.components(separatedBy: patterns)
        let cleanedString = separatedStrings.joined()
        return cleanedString
    }
    
}

extension String {
    func toDictionary() -> [String: Any]? {
        if let data = self.data(using: .utf8) {
            do {
                let json = try JSONSerialization.jsonObject(with: data, options: .mutableContainers) as? [String: Any]
                return json
            } catch {
                print("Something went wrong")
            }
        }
        return nil
    }
}
