//
//  File.swift
//  videochat_ios_demo
//
//  Created by George Mo on 2024/8/21.
//

import UIKit
import videochat_ios_sdk

class VideoChatViewController: UIViewController {
    
//    private var params: TYRTCAuthParams
//    private var avatarInitData :TYAvatarInitData
//    private let config: TYRequestConfig?
    let videoView = UIView()
    
    let subtitleTableView = UITableView()
    private var subtitles: [TYVoiceChatMessage] = []
    
    let stateLabel = UILabel()
    lazy var timerLabel: UILabel = {
        let l = UILabel()
        l.text = "00:00"
        l.textColor = .darkText
        l.font = .systemFont(ofSize: 15)
        return l
    }()
    
    lazy var performanceLabel: UILabel = {
        let l = UILabel()
        l.textColor = .green
        l.numberOfLines = 0
        l.text = """
        本轮对话延时(ms)
        计算中...
        """
        return l
    }()
    lazy var performanceView: UIView = {
        let v = UIView()
        v.backgroundColor = .black.withAlphaComponent(0.4)
        v.addSubview(performanceLabel)
        performanceLabel.snp.makeConstraints { make in
            make.edges.equalToSuperview().inset(6)
        }
        return v
    }()
    
    lazy var push2TalkButton: UIButton = {
        let b = UIButton()
        b.setTitle("长按发送语音", for: .normal)
        b.backgroundColor = UIColor.aly_fromHex(0x49419E)
        b.clipsToBounds = true
        b.layer.cornerRadius = 8
        b.aly_hitEdgeInsets = .init(top: 8, left: 4, bottom: 8, right: 4)
        return b
    }()
    
    var push2TalkButtonLayer: CALayer?
    
    
    lazy var muteButton: UIButton = {
        let b = UIButton()
        b.clipsToBounds = true
        b.backgroundColor = UIColor.aly_fromHex(0x49419E)
        b.aly_hitEdgeInsets = .init(top: 4, left: 4, bottom: 4, right: 4)
        b.setImage(UIImage(named: "voiceChat_voice_input_white"), for: .normal)
        b.setImage(UIImage(named: "voiceChat_voice_mute_white"), for: .selected)
        return b
    }()
    
    private lazy var logButton: UIButton = {
        let button = UIButton()
        button.setTitle("开启日志", for: .normal)
        button.setTitle("导出日志", for: .selected)
        button.titleLabel?.font = .systemFont(ofSize: 15)
        button.setTitleColor(UIColor.aly_fromHex(0x49419E), for: [])
        button.addTarget(self, action: #selector(logButtonTapped), for: .touchUpInside)
        return button
    }()
    
    private lazy var subtitleButton: UIButton = {
        let b = UIButton()
        b.clipsToBounds = true
        b.layer.borderColor = UIColor.aly_fromHex(0x49419E).cgColor
        b.layer.borderWidth = 2
        b.aly_hitEdgeInsets = .init(top: 4, left: 4, bottom: 4, right: 4)
        b.setImage(UIImage(named: "voiceChat_function_subtitleOn"), for: .normal)
        b.setImage(UIImage(named: "voiceChat_function_subtitleOff"), for: .selected)
        b.addTarget(self, action: #selector(subtitleButtonTapped), for: .touchUpInside)
        return b
    }()
    
    /// 返回按钮
    private lazy var hangupButton: UIButton = {
        let button = UIButton()
        button.setImage(.init(named: "left_arrow"), for: [])
        button.aly_hitEdgeInsets = .init(top: -16, left: -16, bottom: -16, right: -16)
        button.addTarget(self, action: #selector(hangupButtonTapped), for: .touchUpInside)
        return button
    }()
    var videoChat: TYVideoChat
    let mode: TYVoiceChatMode
    
    init(license: String, avatarInitData: TYAvatarInitData, dialogConfig: TYDialogConfig) {
        self.mode = dialogConfig.mode
//        self.videoChat = TYVideoChat(activity: videoView, license: license, avatarInitData: avatarInitData, dialogConfig: dialogConfig)
        self.videoChat = TYVideoChat(license: license, avatarInitData: avatarInitData, dialogConfig: dialogConfig)
        let rtcConfig = self.videoChat.getRtcConfig()
        rtcConfig.playoutVolume = 100
        rtcConfig.recordAudioVolumeBeforeVAD = mode == .duplex ? 40 : 100
        rtcConfig.renderMode = .fill
        super.init(nibName: nil, bundle: nil)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view.
        UIApplication.shared.isIdleTimerDisabled = true
        view.backgroundColor = .white
        view.addSubview(videoView)
        view.addSubview(subtitleTableView)
        view.addSubview(stateLabel)
        view.addSubview(timerLabel)
        view.addSubview(hangupButton)
        view.addSubview(push2TalkButton)
        view.addSubview(muteButton)
        view.addSubview(subtitleButton)
        view.addSubview(logButton)
        view.addSubview(performanceView)
        videoView.backgroundColor = .white
        stateLabel.textColor = .black
        stateLabel.textAlignment = .center
        stateLabel.text = ""
        subtitleTableView.backgroundColor = .black.withAlphaComponent(0.1)
        subtitleTableView.dataSource = self
        subtitleTableView.register(SubtitleTableViewCell.self, forCellReuseIdentifier: "subtitle")
        subtitleTableView.rowHeight = UITableView.automaticDimension
        subtitleTableView.estimatedRowHeight = 60
        subtitleTableView.separatorStyle = .none
        
        hangupButton.snp.makeConstraints { make in
            make.leading.equalToSuperview().inset(20)
            make.top.equalTo(self.view.safeAreaLayoutGuide.snp.top).inset(10)
            
        }
        timerLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.centerY.equalTo(self.hangupButton.snp.centerY)
        }
        
        
        logButton.snp.makeConstraints { make in
            make.trailing.equalToSuperview().inset(20)
            make.top.equalTo(self.view.safeAreaLayoutGuide.snp.top).inset(10)
        }
        
        performanceView.snp.makeConstraints { make in
            make.top.equalTo(hangupButton.snp.bottom).offset(8)
            make.leading.equalToSuperview().inset(20)
        }
        
        videoView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        subtitleTableView.snp.makeConstraints { make in
            make.bottom.equalTo(stateLabel.snp.top).offset(-10)
            make.height.equalTo(self.videoView.snp.height).multipliedBy(0.4)
            make.leading.trailing.equalToSuperview().inset(20)
        }
        
        stateLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.height.equalTo(30)
            make.bottom.equalTo(self.push2TalkButton.snp.top).offset(-20)
        }
        
        push2TalkButton.snp.makeConstraints { make in
            make.bottom.equalTo(self.view.safeAreaLayoutGuide.snp.bottom).inset(20)
            make.centerX.equalToSuperview()
            make.height.equalTo(48)
            make.width.equalTo(200)
        }
        muteButton.snp.makeConstraints { make in
            make.center.equalTo(push2TalkButton.snp.center)
            make.height.width.equalTo(48)
        }
        muteButton.layer.cornerRadius = 24
        subtitleButton.snp.makeConstraints { make in
            make.centerY.equalTo(muteButton.snp.centerY)
            make.leading.equalToSuperview().inset(20)
            make.height.width.equalTo(36)
        }
        subtitleButton.layer.cornerRadius = 18
        self.performanceView.isHidden = true
        self.performanceLabel.isHidden = true
        self.push2TalkButton.isHidden = self.mode != .push2talk
        self.muteButton.isHidden = self.mode == .push2talk
        
        //        let layer = aly_createGradientLayer(startColor: UIColor.aly_fromHex(0x5E3579),
        //                                            endColor: UIColor.aly_fromHex(0x49419E),
        //                                            startPoint: CGPoint(x: 0.5, y: 0),
        //                                            endPoint: CGPoint(x: 0.5, y: 1))
        //        push2TalkButton.layer.addSublayer(layer)
        //        layer.frame = push2TalkButton.bounds
        //        self.push2TalkButtonLayer = layer
        
        videoView.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(videoViewTapped)))
        push2TalkButton.addGestureRecognizer(UILongPressGestureRecognizer(target: self, action: #selector(push2TalkButtonLongPressed)))
        muteButton.addTarget(self, action: #selector(muteButtonTapped), for: .touchUpInside)
        
       
        
        videoChat.onMessageReceived = { msg in
            DispatchQueue.main.async {
                let text = "\(msg.type == .speaking ? "我" : "AI" ): \(msg.text ?? "")"
                //                ALYToast.shared.showToast(text: text, style: .info)
            }
            if msg.isFinished {
                self.subtitles.append(msg)
                
                DispatchQueue.main.async {
                    self.subtitleTableView.reloadData()
                    
                    self.subtitleTableView.scrollToRow(at: IndexPath(row: self.subtitles.count - 1, section: 0),
                                                       at: .top, animated: true)
                }
            }
        }
        
        //        Timer.scheduledTimer(withTimeInterval: 1, repeats: true) { t in
        //            let r = Int(Date().timeIntervalSinceReferenceDate) % 2
        //            let msg = TYVoiceChatMessage(type: TYVoiceChatMessageType(rawValue: r) ?? .speaking,
        //                                         text: "no object with given ID \(r)" + (r == 0 ? "" : "\n text \n text"),
        //                                         isFinished: true)
        //            self.videoChat.onMessageReceived?(msg)
        //        }
        
        videoChat.onChatTimerChanged = { seconds in
            let timeString = ALYDateFormatter.formatHourMinSec(time: seconds)
            DispatchQueue.main.async {
                self.timerLabel.text = timeString
            }
        }
        videoChat.onStateChanged = { [weak self] state in
            guard let self else { return }
            DispatchQueue.main.async {
                switch state {
                case .idle:
                    if self.mode == .push2talk {
                        self.stateLabel.text = ""
                    } else {
                        self.stateLabel.text = "连接中..."
                    }
                    break
                case .listening:
                    self.stateLabel.text = "你说，我正在听"
                    break
                case .responding:
                    self.stateLabel.text = "我正在说，轻点可打断我"
                    break
                case .thinking:
                    self.stateLabel.text = "让我想想..."
                    break
                default:
                    break
                }
            }
        }
        
        videoChat.onErrorReceived = { [weak self] error in
            DispatchQueue.main.async {
                ALYToast.shared.showToast(text:String(error.code) + ": " + (error.message ?? "系统错误，请重新加入"))
                self?.videoChat.exit()
            }
        }
        
        videoChat.onPerformaceInfoTrack = { [weak self] type,performanceInfo in
            
            do {
                if let jsonData = performanceInfo.data(using: .utf8) {
                    let jsonObject = try JSONSerialization.jsonObject(with: jsonData, options: [])
                    if let payLoad = jsonObject as? [String: Any] {
                        var llm: Int64 = -1
                        if let llmFirstTokenTime = payLoad["llmFirstTokenTime"] as? String,
                           let llmRequestTime = payLoad["llmRequestTime"] as? String {
                            llm = (Int64(llmFirstTokenTime) ?? 0) - (Int64(llmRequestTime) ?? 0)
                        }
                        var tts: Int64 = -1
                        if let llmFirstTokenTime = payLoad["llmFirstTokenTime"] as? String,
                           let ttsFirstByteTime = payLoad["ttsFirstByteTime"] as? String {
                            tts = (Int64(ttsFirstByteTime) ?? 0) - (Int64(llmFirstTokenTime) ?? 0)
                        }
                        var avatar: Int64 = -1
                        if let respondingStartTime = payLoad["respondingStartTime"] as? String,
                           let ttsFirstByteTime = payLoad["ttsFirstByteTime"] as? String {
                            avatar = (Int64(respondingStartTime) ?? 0) - (Int64(ttsFirstByteTime) ?? 0)
                        }
                        let infoText = """
                        本轮对话延时(ms)
                        LLM: \(llm)
                        TTS: \(tts)
                        Avatar: \(avatar)
                        """
                        DispatchQueue.main.async {
                            self?.performanceLabel.text = infoText
                            ALYToast.shared.showToast(text: infoText, style: .success)
                        }
                    } else {
                        print("解析后的对象不是一个字典")
                    }
                } else {
                    print("无法将JSON字符串转换为Data")
                }
            } catch {
                print("解析JSON时发生错误: \(error)")
            }
        }
    
        
        self.stateLabel.text = "连接中..."
        videoChat.start { isSuccess, error in
               if !isSuccess{
                    print(error?.localizedDescription)
                    DispatchQueue.main.async {
                    if !isSuccess {
                        ALYToast.shared.showToast(text: error?.message ?? "系统错误，请重新加入", style: .error)
                    }
                }
            }
        }
        videoChat.setupRTCView(videoView)
    }
    
    override func viewWillTransition(to size: CGSize, with coordinator: UIViewControllerTransitionCoordinator) {
        super.viewWillTransition(to: size, with: coordinator)
        coordinator.animate { context in
            self.setupButtonLayout()
        }
    }
    
    func setupButtonLayout() {
        self.push2TalkButtonLayer?.frame = self.push2TalkButton.bounds
    }
    
    @objc func videoViewTapped() {
        videoChat.interrupt()
        
    }
    
    @objc func hangupButtonTapped() {
        self.navigationController?.popViewController(animated: true)
        videoChat.exit()
    }
    
    @objc func push2TalkButtonLongPressed(gs: UILongPressGestureRecognizer) {
        switch gs.state {
        case .possible:
            break
        case .began:
            self.videoChat.startSpeech()
            self.push2TalkButton.backgroundColor = UIColor.aly_fromHex(0x5E3579)
            self.push2TalkButton.setTitle("说话中...松开发送", for: .normal)
            break
        case .changed:
            break
        case .ended:
            self.videoChat.stopSpeech()
            self.push2TalkButton.setTitle("长按发送语音", for: .normal)
            self.push2TalkButton.backgroundColor = UIColor.aly_fromHex(0x49419E)
            break
        case .cancelled:
            print("long press cancelled")
            break
        case .failed:
            print("long press failed")
            break
        case .recognized:
            break
        @unknown default:
            break
        }
    }
    
    @objc func logButtonTapped(sender: UIButton) {
        let isStarted = sender.isSelected // 当前状态
        if isStarted { // 日志已开启
            self.videoChat.debugExportLogs(self) {
                DispatchQueue.main.async {
                    ALYToast.shared.showToast(text: "日志已导出", style: .success)
                }
            }
        } else { // 日志未开启
            self.videoChat.debugStartLogs()
            ALYToast.shared.showToast(text: "日志已开启", style: .success)
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
            sender.isSelected = !isStarted
        }
    }
    
    @objc func muteButtonTapped(sender: UIButton) {
        sender.isSelected = !sender.isSelected
        let isMuted = sender.isSelected
        self.videoChat.muteLocalMic(mute: isMuted)
        if isMuted {
            ALYToast.shared.showToast(text: "麦克风已静音", style: .info)
        }
    }
    
    @objc func subtitleButtonTapped(sender: UIButton) {
        sender.isSelected = !sender.isSelected
        self.subtitleTableView.isHidden = sender.isSelected
    }
}


extension VideoChatViewController: UITableViewDataSource {
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return subtitles.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "subtitle") as! SubtitleTableViewCell
        
        let subtitle = subtitles[indexPath.row]
        cell.subtitleLabel.text = subtitle.text
        if subtitle.type == .responding {
            cell.iconView.image = UIImage(named: "tongyi_logo")
            cell.iconView.isHidden = false
            cell.updateLayout(left: 12, right: 48, iconWidth: 24)
        } else {
            cell.iconView.image = nil
            cell.iconView.isHidden = true
            cell.updateLayout(left: 48, right: 12, iconWidth: 0)
        }
        return cell
    }
}

class SubtitleTableViewCell: UITableViewCell {
    
    let iconView = UIImageView()
    let subtitleLabel = UILabel()
    let mainView = UIView()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        
        createView()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func createView() {
        self.backgroundColor = .clear
        
        mainView.backgroundColor = .white
        mainView.clipsToBounds = true
        mainView.layer.cornerRadius = 6
        
        contentView.addSubview(mainView)
        mainView.snp.makeConstraints { make in
            make.leading.trailing.equalToSuperview().inset(8)
            make.top.bottom.equalToSuperview().inset(4)
        }
        mainView.addSubview(iconView)
        mainView.addSubview(subtitleLabel)
        iconView.snp.makeConstraints { make in
            make.leading.equalToSuperview().inset(8)
            make.top.equalToSuperview().inset(8)
            make.height.equalTo(24)
            make.width.equalTo(24)
        }
        subtitleLabel.snp.makeConstraints { make in
            make.leading.equalTo(iconView.snp.trailing).offset(12)
            make.top.bottom.trailing.equalToSuperview().inset(4)
        }
        subtitleLabel.numberOfLines = 0
        subtitleLabel.textAlignment = .left
        subtitleLabel.textColor = .black
        subtitleLabel.font = .systemFont(ofSize: 14)
        iconView.contentMode = .scaleAspectFit
    }
    
    func updateLayout(left: Int, right: Int, iconWidth: Int) {
        mainView.snp.updateConstraints { make in
            make.leading.equalToSuperview().inset(left)
            make.trailing.equalToSuperview().inset(right)
        }
        iconView.snp.updateConstraints { make in
            make.width.equalTo(iconWidth)
        }
    }
}
