//
//  FileTool.swift
//  videochat_ios_demo
//
//  Created by FlyingFish on 2025/8/20.
//

import Foundation

public enum FileManagerError: Error {
    case bundleFolderNotFound
    case documentsDirectoryUnavailable
    case createDirectoryFailed
    case copyFailed(description: String)
}

public class FileTool {
    public static func copyFolderFromBundleToDocuments(
            srcDirURL: URL,
            dstDirURL: URL,
            overwrite: Bool = true
        ) throws {
            let fileManager = FileManager.default
            
            // 创建目标根目录
            do {
                try fileManager.createDirectory(
                    at: dstDirURL,
                    withIntermediateDirectories: true
                )
            } catch {
                throw FileManagerError.createDirectoryFailed
            }
            
            // mlpackage单独处理
            let mlpackages = ["Codec.mlpackage", "a2bs.mlpackage"]
            for mlpackage in mlpackages {
                let srcMlPackage = srcDirURL.appendingPathComponent(mlpackage)
                let dstMlPackage = dstDirURL.appendingPathComponent(mlpackage)
                // 处理文件覆盖
                if overwrite && fileManager.fileExists(atPath: dstMlPackage.path) {
                    try fileManager.removeItem(at: dstMlPackage)
                }
                
                // 执行拷贝
                do {
                    print("copy srcMlPackage: \(srcMlPackage.absoluteString) \n dstMlPackage: \(dstMlPackage.absoluteString)")
                    try fileManager.copyItem(at: srcMlPackage, to: dstMlPackage)
                } catch {
                    throw FileManagerError.copyFailed(
                        description: "文件拷贝失败: \(srcMlPackage.lastPathComponent)\n\(error.localizedDescription)"
                    )
                }
            }
            
            
            // 递归遍历源目录
            guard let enumerator = fileManager.enumerator(
                at: srcDirURL,
                includingPropertiesForKeys: [.isDirectoryKey],
                options: [.skipsHiddenFiles]
            ) else {
                throw FileManagerError.copyFailed(description: "无法遍历 Bundle 目录")
            }
            
            for case let sourceFileURL as URL in enumerator {
                // 获取相对路径
                let relativePath = sourceFileURL
                    .relativePath
                    .replacingOccurrences(
                        of: srcDirURL.relativePath,
                        with: ""
                    )
                if (relativePath.contains(".mlpackage")) {
                    print("ml package skip copy")
                    continue;
                }
                
                let destinationFileURL = dstDirURL
                    .appendingPathComponent(relativePath)
                
                // 获取文件类型
                let resourceValues = try sourceFileURL.resourceValues(forKeys: [.isDirectoryKey])
                let isDirectory = resourceValues.isDirectory ?? false
                
                if isDirectory {
                    // 创建子目录
                    try fileManager.createDirectory(
                        at: destinationFileURL,
                        withIntermediateDirectories: true
                    )
                } else {
                    // 处理文件覆盖
                    if overwrite && fileManager.fileExists(atPath: destinationFileURL.path) {
                        try fileManager.removeItem(at: destinationFileURL)
                    }
                    
                    // 执行拷贝
                    do {
                        if (!sourceFileURL.pathExtension.contains("png") && !sourceFileURL.pathExtension.contains("jpg")) {
                            print("copy file: \(sourceFileURL.absoluteString) to \(destinationFileURL.absoluteString)")
                        }
                        try fileManager.copyItem(at: sourceFileURL, to: destinationFileURL)
                    } catch {
                        throw FileManagerError.copyFailed(
                            description: "文件拷贝失败: \(sourceFileURL.lastPathComponent)\n\(error.localizedDescription)"
                        )
                    }
                }
            }
        }
}
