//
//  TYPermissionManager.swift
//  TongyiApp
//
//  Created by George Mo on 2023/9/19.
//  Copyright © 2023 Aliyun. All rights reserved.
//

import AVFoundation
import Photos

struct Permission {
    let type: TYPermissionKind
    let enabled: Bool
}

class TYPermissionManager: NSObject {
    
    static let shared = TYPermissionManager()
    
    private override init() {
        super.init()
    }
    
    // MARK: - 麦克风权限
    
    /// 验证是否授权了麦克风权限
    /// - Parameters:
    ///   - doCompletionAfterAuthorized: 如果初始状态是 notDetermined， 授权后是否还有执行回调
    ///   - grantedCompletion: 授权后获或已经授权后才执行的回调。是否执行具体见上
    ///   - deniedCancelAction: 授权失败弹窗，用户取消操作回调
    func validateAudioAccess(doCompletionAfterAuthorized: Bool = true,
                             grantedCompletion: @escaping () -> Void,
                             deniedCancelAction: (() -> Void)? = nil) {
        let permission = TYPermission(kind: .microphone)
        switch permission.status {
        case .authorized:
            grantedCompletion()
        case .denied:
            self.showAudioAccessUnAuthorizedAlert(deniedCancelAction)
        case .notDetermined:
            if (doCompletionAfterAuthorized) {
                permission.request { [weak self] in
                    self?.validateAudioAccess(grantedCompletion: grantedCompletion, deniedCancelAction: deniedCancelAction)
                }
            } else {
                permission.request { }
            }
        default:
            break
        }
    }
    
    /// 显示麦克风没有授权的Alert
    func showAudioAccessUnAuthorizedAlert(_ cancelAction: (() -> Void)? = nil) {
        self.showAlert(title: "语音权限未开启", message: "为了能使用语音输入、语音通话等服务，请开启您的麦克风权限", cancelAction: cancelAction)
    }
    
    // MARK: - 相册权限
}

extension TYPermissionManager {
    /// 显示Alert
    /// - Parameters:
    ///   - title: 标题
    ///   - message:
    ///   - cancelAction: 取消按钮的回调
    func showAlert(title: String, message: String, cancelAction: (() -> Void)? = nil) {
        let alert = UIAlertController(title: title,
                                      message: message,
                                      preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "取消", style: .default, handler: { _ in cancelAction?() }))
        alert.addAction(UIAlertAction(title: "去设置", style: .cancel, handler: { _ in
            TYPermissionManager.shared.gotoAppSetting()
        }))
        aly_topmostViewController()?.present(alert, animated: true)
    }
    
    /// 去app系统设置
    func gotoAppSetting() {
        guard let settingsURL = URL(string: UIApplication.openSettingsURLString) else { return }
        if UIApplication.shared.canOpenURL(settingsURL) {
            UIApplication.shared.open(settingsURL)
        }
    }
}
