//
//  UIColor+ALYExt.swift
//  TongyiApp
//
//  Created by Wien on 2023/9/8.
//  Copyright © 2023 Aliyun. All rights reserved.
//

import UIKit

extension UIColor {
    
    public static func aly_fromHex(_ rgbValue:UInt32, alpha:CGFloat = 1.0) -> UIColor {
        let red = CGFloat((rgbValue & 0xFF0000) >> 16) / 256.0
        let green = CGFloat((rgbValue & 0xFF00) >> 8) / 256.0
        let blue = CGFloat(rgbValue & 0xFF) / 256.0
        return UIColor(red:red, green:green, blue:blue, alpha:alpha)
    }
    
    public static func aly_fromARGB(_ argbValue: UInt32) -> UIColor {
        let alpha   = CGFloat(argbValue >> 24 & 0xff) / 256.0
        let red     = CGFloat(argbValue >> 16 & 0xff) / 256.0
        let green   = CGFloat(argbValue >> 8 & 0xff) / 256.0
        let blue    = CGFloat(argbValue & 0xff) / 256.0
        return UIColor(red:red, green:green, blue:blue, alpha:alpha)
    }
    
    public static func aly_fromHexString(_ hexString: String, alpha: CGFloat = 1.0) -> UIColor? {
        // AARRGGBB
        if hexString.count > 7 {
            return self.aly_fromARGBString(hexString)
        }
        // RRGGBB
        var hexSanitized = hexString.trimmingCharacters(in: .whitespacesAndNewlines)
        hexSanitized = hexSanitized.replacingOccurrences(of: "#", with: "")
        
        if let rgb = UInt32(hexSanitized, radix: 16) {
            return aly_fromHex(rgb, alpha: alpha)
        } else {
            return nil
        }
    }
    
    private static func aly_fromARGBString(_ hexString: String) -> UIColor? {
        var hexSanitized = hexString.trimmingCharacters(in: .whitespacesAndNewlines)
        hexSanitized = hexSanitized.replacingOccurrences(of: "#", with: "")
        
        if let argb = UInt32(hexSanitized, radix: 16) {
            return aly_fromARGB(argb)
        } else {
            return nil
        }
    }
    
    public static func aly_dynamicColor(light: UIColor, dark: UIColor) -> UIColor {
        var color: UIColor
        if #available(iOS 13.0, *) {
            color = UIColor { $0.userInterfaceStyle == .dark ? dark : light }
        } else {
            color = light
        }
        return color
    }
    
}

extension UIColor {
    
    /// 提取并返回颜色的红色（Red）分量。
    public var aly_red: CGFloat {
        var r: CGFloat = 0 // 定义一个变量来存储红色分量的值
        self.getRed(&r, green: nil, blue: nil, alpha: nil) // 获取当前颜色的红色分量
        return r // 返回红色分量的值
    }
    
    /// 提取并返回颜色的绿色（Green）分量。
    public var aly_green: CGFloat {
        var g: CGFloat = 0 // 定义一个变量来存储绿色分量的值
        self.getRed(nil, green: &g, blue: nil, alpha: nil) // 获取当前颜色的绿色分量
        return g // 返回绿色分量的值
    }
    
    /// 提取并返回颜色的蓝色（Blue）分量。
    public var aly_blue: CGFloat {
        var b: CGFloat = 0 // 定义一个变量来存储蓝色分量的值
        self.getRed(nil, green: nil, blue: &b, alpha: nil) // 获取当前颜色的蓝色分量
        return b // 返回蓝色分量的值
    }
    
    /// 提取并返回颜色的透明度（Alpha）分量。
    public var aly_alpha: CGFloat {
        return self.cgColor.alpha // 直接从颜色的 CGColor 获取透明度分量并返回它
    }
    
}


/// 创建渐变layer
/// - Parameters:
///   - startColor: 开始点颜色
///   - endColor: 结束点颜色
///   - startPoint: 开始点
///   - endPoint: 结束点
public func aly_createGradientLayer(startColor: UIColor = .aly_fromHex(0x615CED), endColor: UIColor = .aly_fromHex(0x534AD1), startPoint: CGPoint = CGPoint(x: 0, y: 0.5), endPoint: CGPoint = CGPoint(x: 1, y: 0.5)) -> CAGradientLayer {
    let layer = CAGradientLayer()
    layer.colors = [startColor.cgColor, endColor.cgColor]
    layer.locations = [0, 1]
    layer.startPoint = startPoint
    layer.endPoint = endPoint
    layer.frame = .zero
    return layer
}

public func aly_createImage(_ color: UIColor, size: CGSize = CGSize(width: 1.0, height: 1.0)) -> UIImage? {
    let rect = CGRect(origin: .zero, size: size)
    UIGraphicsBeginImageContext(size)
    let context = UIGraphicsGetCurrentContext()
    context?.setFillColor(color.cgColor)
    context?.fill([rect])
    
    let image = UIGraphicsGetImageFromCurrentImageContext()
    UIGraphicsEndImageContext()
    return image
}
