//
//  File.swift
//  videochat_ios_demo
//
//  Created by George Mo on 2024/8/21.
//

import UIKit
import digital_human_ios_sdk
import SnapKit



class VideoChatViewController: UIViewController {

    let avatarView = UIView()
    let avatarContainerView = UIView()
    let avatarInitData: TYAvatarInitData
    let dialogConfig: TYDialogConfig
    let subtitleTableView = UITableView()
    
    var chatState : TYVoiceChatState?
    
    private let audioDataQueue = DispatchQueue(
        label: "com.audio.data.sync",
        qos: .userInitiated,
        attributes: [.concurrent],
        autoreleaseFrequency: .workItem
    )
    
    private var _audioData = Data() // 私有存储
    private var audioData: Data {
        get { audioDataQueue.sync { _audioData } }
        set { audioDataQueue.async(flags: .barrier) { self._audioData = newValue } }
    }
    
    private var record = TYAudioRecorder()
    private var subtitles: [TYVoiceChatMessage] = []

    lazy var stateLabel: UILabel = {
        let l = UILabel()
        l.textColor = .white
        l.font = .systemFont(ofSize: 16, weight: .bold)
        l.textAlignment = .center
        return l
    }()

    lazy var timerLabel: UILabel = {
        let l = UILabel()
        l.text = "00:00"
        l.textColor = .darkText
        l.font = .systemFont(ofSize: 15)
        return l
    }()

    lazy var performanceLabel: UILabel = {
        let l = UILabel()
        l.textColor = .green
        l.numberOfLines = 0
        l.font = .systemFont(ofSize: 14)
        l.text = """
        本轮对话延时(ms)
        计算中...
        """
        return l
    }()
    lazy var frameIdLabel: UILabel = {
        let l = UILabel()
        l.text = ""
        l.textColor = .lightGray
        l.font = .systemFont(ofSize: 14)
        return l
    }()
    lazy var fpsLabel: UILabel = {
        let l = UILabel()
        l.text = "FPS:0"
        l.textColor = .lightGray
        l.font = .systemFont(ofSize: 14)
        return l
    }()
    lazy var performanceView: UIView = {
        let v = UIView()
        v.backgroundColor = .black.withAlphaComponent(0.4)
        v.addSubview(performanceLabel)
        performanceLabel.snp.makeConstraints { make in
            make.edges.equalToSuperview().inset(6)
        }
        v.clipsToBounds = true
        v.layer.cornerRadius = 4
        return v
    }()

    lazy var push2TalkButton: UIButton = {
        let b = UIButton()
        b.setTitle("长按发送语音", for: .normal)
        b.backgroundColor = UIColor.aly_fromHex(0x49419E)
        b.clipsToBounds = true
        b.layer.cornerRadius = 8
        b.aly_hitEdgeInsets = .init(top: 8, left: 4, bottom: 8, right: 4)
        return b
    }()

    var push2TalkButtonLayer: CALayer?


    lazy var muteButton: UIButton = {
        let b = UIButton()
        b.clipsToBounds = true
        b.backgroundColor = .white.withAlphaComponent(0.2)
        b.aly_hitEdgeInsets = .init(top: 4, left: 4, bottom: 4, right: 4)
        b.setImage(UIImage(named: "voiceChat_voice_input_white"), for: .normal)
        b.setImage(UIImage(named: "voiceChat_voice_mute_white"), for: .selected)
        b.addTarget(self, action: #selector(muteButtonTapped), for: .touchUpInside)
        return b
    }()

    private lazy var logButton: UIButton = {
        let button = UIButton()
        button.setTitle("导出音频", for: .normal)
        button.titleLabel?.font = .systemFont(ofSize: 15)
        button.setTitleColor(UIColor.aly_fromHex(0x49419E), for: [])
        button.addTarget(self, action: #selector(logButtonTapped), for: .touchUpInside)
//        let doubleTap = UITapGestureRecognizer(target: self, action: #selector(logButtonDoubleTapped))
//        doubleTap.numberOfTapsRequired = 2
//        button.addGestureRecognizer(doubleTap)
        return button
    }()

    private lazy var toggleRTButton: UIButton = {
        let button = UIButton()
        button.setTitle("显示RT", for: .normal)
        button.setTitle("关闭RT", for: .selected)
        button.titleLabel?.font = .systemFont(ofSize: 15)
        button.setTitleColor(UIColor.aly_fromHex(0x49419E), for: [])
        button.addTarget(self, action: #selector(toggleRTButtonTapped), for: .touchUpInside)
        return button
    }()

    private lazy var testAudioButton: UIButton = {
        let button = UIButton()
        button.setTitle("测试音频驱动", for: .normal)
        //button.setTitle("关闭RT", for: .selected)
        button.titleLabel?.font = .systemFont(ofSize: 15)
        button.setTitleColor(UIColor.aly_fromHex(0x49419E), for: [])
        button.addTarget(self, action: #selector(testAudioDataButtonTapped), for: .touchUpInside)
        return button
    }()
    

    private lazy var subtitleButton: UIButton = {
        let b = UIButton()
        b.clipsToBounds = true
        b.aly_hitEdgeInsets = .init(top: 4, left: 4, bottom: 4, right: 4)
        b.backgroundColor = .white.withAlphaComponent(0.2)
        b.setImage(UIImage(named: "voiceChat_function_subtitleOn"), for: .normal)
        b.setImage(UIImage(named: "voiceChat_function_subtitleOff"), for: .selected)
        b.addTarget(self, action: #selector(subtitleButtonTapped), for: .touchUpInside)
        return b
    }()

    /// 返回按钮
    private lazy var hangupButton: UIButton = {
        let button = UIButton()
        button.setImage(.init(named: "left_arrow"), for: [])
        button.addTarget(self, action: #selector(hangupButtonTapped), for: .touchUpInside)
        return button
    }()
    let loadingHub = MBProgressHUD(frame: .zero)

    // 自动打断任务
    private var autoInterruptTimer: Timer?
    private let INVOKE_INTERRUPT_TIMEOUT: Double = 5.0 // 思考超时5秒
    private let INVOKE_INTERRUPT_TIMEOUT_FOR_AUDIO_LAG: Double = 2.0 // AudioLag超时2秒

    private var needRequestPermission = false
    let videoChat: TYVideoChat
    let mode: TYVoiceChatMode
    var localAvatarRenderView: UIView? // 端渲染数字人的渲染组件


    init(license: String, avatarInitData: TYAvatarInitData, dialogConfig: TYDialogConfig) {
        self.mode = dialogConfig.mode
        self.avatarInitData = avatarInitData
        self.dialogConfig = dialogConfig
        self.videoChat = TYVideoChat(license: license, avatarInitData: avatarInitData, dialogConfig: dialogConfig)
        let rtcConfig = self.videoChat.getRtcConfig()
        rtcConfig.playoutVolume = 100
        rtcConfig.renderMode = .crop
        super.init(nibName: nil, bundle: nil)
        if self.dialogConfig.renderType == .avatar_only {
            record.setup(sampleRate:Int32(dialogConfig.outboundSampleRate) , numOfChannels: 1, bitsPerChannel: 16)
            record.delegate = self
        }
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    deinit {
        TYLogger.log("deinit vc.")
        NotificationCenter.default.removeObserver(self)
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view.
        UIApplication.shared.isIdleTimerDisabled = true
        
        //        view.backgroundColor = .gray
        
        self.addNotifications()
        
        self.view.addSubview(avatarContainerView)
        
        view.addSubview(subtitleTableView)
        view.addSubview(stateLabel)
        view.addSubview(timerLabel)
        view.addSubview(hangupButton)
        view.addSubview(push2TalkButton)
        view.addSubview(muteButton)
        view.addSubview(subtitleButton)
        view.addSubview(logButton)
        view.addSubview(toggleRTButton)
        view.addSubview(performanceView)
        view.addSubview(loadingHub)
        
        subtitleTableView.backgroundColor = .black.withAlphaComponent(0.1)
        subtitleTableView.dataSource = self
        subtitleTableView.register(SubtitleTableViewCell.self, forCellReuseIdentifier: "subtitle")
        subtitleTableView.rowHeight = UITableView.automaticDimension
        subtitleTableView.estimatedRowHeight = 60
        subtitleTableView.separatorStyle = .none
        
        avatarContainerView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        if self.dialogConfig.renderType == .cloud {
            self.avatarView.backgroundColor = .white
            self.avatarContainerView.addSubview(self.avatarView)
            self.avatarView.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(containerViewTapped)))
            self.videoChat.setupRTCView(self.avatarView)
            self.avatarView.snp.makeConstraints { make in
                make.edges.equalToSuperview()
            }
        }
        
        hangupButton.snp.makeConstraints { make in
            make.leading.equalToSuperview().inset(20)
            make.top.equalTo(self.view.safeAreaLayoutGuide.snp.top).inset(10)
            
        }
        timerLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.centerY.equalTo(self.hangupButton.snp.centerY)
        }
        
        
        logButton.snp.makeConstraints { make in
            make.trailing.equalToSuperview().inset(20)
            make.top.equalTo(self.view.safeAreaLayoutGuide.snp.top).inset(10)
        }
        toggleRTButton.snp.makeConstraints { make in
            make.trailing.equalToSuperview().inset(20)
            make.top.equalTo(logButton.snp.bottom).offset(0)
        }
        if self.dialogConfig.renderType == .avatar_only {
            view.addSubview(testAudioButton)
            testAudioButton.snp.makeConstraints { make in
                make.trailing.equalToSuperview().inset(20)
                make.top.equalTo(toggleRTButton.snp.bottom).offset(0)
            }
            performanceView.snp.makeConstraints { make in
                make.top.equalTo(testAudioButton.snp.bottom).offset(4)
                make.trailing.equalToSuperview().inset(20)
            }
        }else {
            performanceView.snp.makeConstraints { make in
                make.top.equalTo(toggleRTButton.snp.bottom).offset(4)
                make.trailing.equalToSuperview().inset(20)
            }
        }
        
        subtitleTableView.snp.makeConstraints { make in
            make.bottom.equalTo(stateLabel.snp.top).offset(-20)
            make.height.equalTo(self.view.snp.height).multipliedBy(0.3)
            make.leading.trailing.equalToSuperview().inset(20)
        }
        
        view.addSubview(fpsLabel)
        fpsLabel.snp.makeConstraints { make in
            make.trailing.bottom.equalTo(subtitleTableView).inset(4)
        }
        view.addSubview(frameIdLabel)
        frameIdLabel.snp.makeConstraints { make in
            make.trailing.equalTo(subtitleTableView).inset(4)
            make.bottom.equalTo(fpsLabel.snp.top).offset(-4)
        }
        frameIdLabel.isHidden = true
        performanceView.isHidden = true
        
        if self.mode == .push2talk || self.dialogConfig.renderType == .avatar_only {
            stateLabel.snp.makeConstraints { make in
                make.centerX.equalToSuperview()
                make.height.equalTo(30)
                make.bottom.equalTo(self.push2TalkButton.snp.top).offset(-20)
            }

            push2TalkButton.snp.makeConstraints { make in
                make.bottom.equalTo(self.view.safeAreaLayoutGuide.snp.bottom).inset(20)
                make.centerX.equalToSuperview()
                make.height.equalTo(48)
                make.width.equalTo(200)
            }
        } else {
            stateLabel.snp.makeConstraints { make in
                make.centerX.equalToSuperview()
                make.height.equalTo(30)
                make.bottom.equalTo(self.view.safeAreaLayoutGuide.snp.bottom).inset(20)
            }
        }
        muteButton.snp.makeConstraints { make in
            make.centerY.equalTo(stateLabel.snp.centerY)
            make.leading.equalToSuperview().inset(20)
            make.height.width.equalTo(48)
        }

        muteButton.layer.cornerRadius = 24
        subtitleButton.snp.makeConstraints { make in
            make.centerY.equalTo(muteButton.snp.centerY)
            make.trailing.equalToSuperview().inset(20)
            make.height.width.equalTo(48)
        }
        subtitleButton.layer.cornerRadius = 24

        self.push2TalkButton.isHidden = (self.mode != .push2talk && self.dialogConfig.renderType != .avatar_only)
        self.muteButton.isHidden = (self.mode == .push2talk || self.dialogConfig.renderType != .avatar_only)

        push2TalkButton.addGestureRecognizer(UILongPressGestureRecognizer(target: self, action: #selector(push2TalkButtonLongPressed)))

        /// 处理无权限或者未获取过权限情况
        self.handleAudioPermission()

        videoChat.onMessageReceived = { [weak self] msg in
            guard let self else { return }
            DispatchQueue.main.async {
                let text = "\(msg.type == .speaking ? "我" : "AI" ): \(msg.text ?? "")"
//                ALYToast.shared.showToast(text: text, style: .info)
            }
            if msg.isFinished {
                self.subtitles.append(msg)

                DispatchQueue.main.async {
                    self.subtitleTableView.reloadData()

                    self.subtitleTableView.scrollToRow(at: IndexPath(row: self.subtitles.count - 1, section: 0),
                                                       at: .top, animated: true)
                }
            }
        }

        videoChat.onChatTimerChanged = { [weak self] seconds in
            let timeString = ALYDateFormatter.formatHourMinSec(time: seconds)
            DispatchQueue.main.async {
                self?.timerLabel.text = timeString
            }
        }
        videoChat.onStateChanged = { [weak self] state in
            guard let self else { return }
            DispatchQueue.main.async {
                self.autoInterruptTimer?.invalidate()
                self.autoInterruptTimer = nil
                self.chatState = state
                switch state {
                case .idle:
                    if self.mode == .push2talk{
                        self.stateLabel.text = ""
                    }else if self.dialogConfig.renderType == .avatar_only {
                        self.stateLabel.text = ""
                    } else {
                        self.stateLabel.text = "连接中..."
                    }
                case .listening:
                    if self.dialogConfig.renderType == .avatar_only {
                        return
                    }
                    self.stateLabel.text = "你说，我正在听"
                case .responding:
                    if self.dialogConfig.renderType == .avatar_only {
                        return
                    }
                    self.stateLabel.text = "我正在说，轻点可打断我"
                case .thinking:
                    if self.dialogConfig.renderType == .avatar_only {
                        return
                    }
                    self.stateLabel.text = "让我想想..."
                    self.autoInterruptTimer = Timer.scheduledTimer(withTimeInterval: self.INVOKE_INTERRUPT_TIMEOUT, repeats: false) { [weak self] _ in
                        guard let self = self else { return }
                        TYLog("思考超时，已自动切换到听")
                        ALYToast.shared.showToast(text: "思考超时，已自动切换到听", style: .info)
                        self.videoChat.interrupt()
                    }
                }
                if state != .responding, self.dialogConfig.renderType == .local {
                    self.frameIdLabel.text = ""
                }
            }
        }

        if self.dialogConfig.renderType != .cloud {
            loadingHub.label.text = "模型准备中..."
            loadingHub.show(animated: true)
        }
        
        videoChat.onLocalAvatarAssetsLoaded = { [weak self] success in
            NSLog("模型资源加载\(success ? "成功": "失败")")
            DispatchQueue.main.async {
                guard let self else { return }
                
                self.loadingHub.hide(animated: true)
                if !success {
                    ALYToast.shared.showToast(text: "模型加载失败，请退出重试", style: .error)
                    return
                }
                
                self.localAvatarRenderView = self.videoChat.addLocalAvatarMetalView(to: self.avatarContainerView)
                self.localAvatarRenderView?.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(self.containerViewTapped)))
                self.localAvatarRenderView?.isHidden = true
                self.stateLabel.text = self.dialogConfig.renderType == .avatar_only ? "" : "连接中..."
                self.videoChat.start { success, error in

                    DispatchQueue.main.async {
                        if !success {
                            ALYToast.shared.showToast(text: error?.message ?? "系统错误，请重新加入", style: .error)
                        }
                    }
                }
            }
        }

        videoChat.localAvatarMotionDataDidLag = { [weak self] in
            DispatchQueue.main.async {
                guard let self = self else { return }
                TYLog("MotionData音频lag触发卡顿",sendNow: true,upload: true)
//                ALYToast.shared.showToast(text: "MotionData音频lag触发卡顿", style: .warning)
                self.autoInterruptTimer?.invalidate()
                self.autoInterruptTimer = Timer.scheduledTimer(
                    withTimeInterval: self.INVOKE_INTERRUPT_TIMEOUT_FOR_AUDIO_LAG,
                    repeats: false
                ) { [weak self] _ in
                    guard let self = self else { return }
                    TYLog("AudioLag超时，已自动切换到听")
                    ALYToast.shared.showToast(text: "AudioLag超时，已自动切换到听", style: .info)
                    self.videoChat.interrupt()
                }
            }
        }
        
        videoChat.onReadyToSpeech = { [weak self] in
            let perFrameTime = 1.0 / 30.0 // 每帧时间间隔，此处等待1帧后再将View渲染出来，避免抠绿场景下出现绿色闪边
            DispatchQueue.main.asyncAfter(deadline: .now() + perFrameTime) {
                self?.localAvatarRenderView?.isHidden = false
            }
        }

        videoChat.onErrorReceived = { [weak self] error in
            DispatchQueue.main.async {
                ALYToast.shared.showToast(text:String(error.code) + ": " + (error.message ?? "系统错误，请重新加入"))
                self?.videoChat.exit()
                self?.autoInterruptTimer?.invalidate()
                self?.autoInterruptTimer = nil
            }
        }

        videoChat.onPerformaceDataReceived = { [weak self] data in
            DispatchQueue.main.async {
                guard let self, self.toggleRTButton.isSelected else {
                    return
                }
                let infoText =
                """
                本轮对话延时(ms)
                llm: \(data.llmDelay)
                tts: \(data.ttsDelay)
                avatar: \(data.avatarDelay)
                """
                self.performanceLabel.text = infoText
                self.performanceView.isHidden = false
                DispatchQueue.main.asyncAfter(deadline: .now() + 5) {
                    self.performanceView.isHidden = true
                }
            }
        }

//        videoChat.localAvatarRenderRTReceived = { timeConst in
//            DispatchQueue.main.async {
//                ALYToast.shared.showToast(text: "端推理渲染延时: \(Int(timeConst * 1000))ms")
//            }
//        }

        videoChat.localAvatarRealtimeFpsCallback = { [weak self] fps in
            DispatchQueue.main.async {
                let displayFPS = String(format: "FPS:%.1f", fps)
                self?.fpsLabel.text = displayFPS
            }
        }

        videoChat.localAvatarFrameInfoCallback = { [weak self] frameInfo in
            DispatchQueue.main.async {
                self?.frameIdLabel.text = frameInfo
            }
        }
        
        videoChat.onInterruptResult = { [weak self] result, errorInfo in
            DispatchQueue.main.async {
                self?.autoInterruptTimer?.invalidate()
                self?.autoInterruptTimer = nil
            }
        }

        // 云渲染，进入页面后即开始加入通道，端渲染/纯端渲染需要等资源加载完成后再加入通道
        if self.dialogConfig.renderType == .cloud {
            self.stateLabel.text = "连接中..."
            videoChat.start { success, error in
                DispatchQueue.main.async {
                    if !success {
                        ALYToast.shared.showToast(text: error?.message ?? "系统错误，请重新加入", style: .error)
                    }
                }
            }
        }
    }

    /// 处理音频权限
    /// - Parameter onFirstAuthorized: 首次授权的回调
    func handleAudioPermission(_ onFirstAuthorized: (() -> Void)? = nil) {
        let permission = TYPermission(kind: .microphone)
        switch permission.status {
        case .authorized:
            break
        case .denied:
            guard !(self.children.last is UIAlertController) else { return }
            TYPermissionManager.shared.showAudioAccessUnAuthorizedAlert {
                DispatchQueue.main.async { [weak self] in
                    self?.popViewController()
                }
            }
        case .notDetermined:
            self.needRequestPermission = true
            permission.request { [weak self] in
                guard let self else { return }
                switch permission.status {
                case .denied:
                    self.navigationController?.popViewController(animated: true)
                case .authorized:
                    onFirstAuthorized?()
                default:
                    break;
                }
            }
        default:
            break
        }
    }


    /// 增加监听事件
    private func addNotifications() {
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(self.onAppBecomeActive),
                                               name: UIApplication.willEnterForegroundNotification,
                                               object: nil)
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(self.onAppResignActive),
                                               name: UIApplication.didEnterBackgroundNotification,
                                               object: nil)
    }

    @objc func onAppBecomeActive() {
        TYLog("App entered foreground.")
        if self.needRequestPermission {
            return
        }
        guard TYPermission(kind: .microphone).status == .authorized else {
            TYPermissionManager.shared.showAudioAccessUnAuthorizedAlert {
                DispatchQueue.main.async { [weak self] in
                    self?.popViewController()
                }
            }
            return
        }
//        self.stateLabel.text = "连接中..."
//        DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) { [weak self] in
//            self?.videoChat.didBeomeActive()
//            self?.videoChat.start { success, error in
//                DispatchQueue.main.async {
//                    if !success {
//                        ALYToast.shared.showToast(text: error?.message ?? "系统错误，请重新加入", style: .error)
//                    } else {
//                        ALYToast.shared.showToast(text: "重连成功", style: .success)
//                    }
//                }
//            }
//        }
    }

    @objc func onAppResignActive() {
        TYLog("App entered background.")
        if self.needRequestPermission {
            return
        }
        // self.videoChat.didEnterBackground()
    }

    private func popViewController() {
        self.navigationController?.popViewController(animated: true)
    }

    override func viewWillTransition(to size: CGSize, with coordinator: UIViewControllerTransitionCoordinator) {
        super.viewWillTransition(to: size, with: coordinator)
        coordinator.animate { context in
            self.setupButtonLayout()
        }
    }

    private func setupButtonLayout() {
        self.push2TalkButtonLayer?.frame = self.push2TalkButton.bounds
    }

    @objc func containerViewTapped() {
        videoChat.interrupt { success in
            print("interrupt finished: \(success)")
        }
    }

    @objc func hangupButtonTapped() {
        self.videoChat.exit()
        self.popViewController()
    }

    @objc func push2TalkButtonLongPressed(gs: UILongPressGestureRecognizer) {
        if self.chatState == .responding {
            self.videoChat.interrupt()
        }
        switch gs.state {
        case .possible:
            break
        case .began:
            if self.mode == .push2talk {
                self.videoChat.startSpeech()
            }else if self.dialogConfig.renderType == .avatar_only {
                self.record.startRecorder()
            }
            self.push2TalkButton.backgroundColor = UIColor.aly_fromHex(0x5E3579)
            self.push2TalkButton.setTitle("说话中...松开发送", for: .normal)
        case .changed:
            break
        case .ended:
            if self.mode == .push2talk {
                self.videoChat.stopSpeech()
            }else if self.dialogConfig.renderType == .avatar_only {
                print("[videoChatViewController] pushAudioData")
                self.record.stopRecorder(shouldNotify: false)
                let pcmSamples = convertToInt16LittleEndian()
                print("pcmSamples \(pcmSamples)")
              _ = self.videoChat.pushAudioData(pcmSamples, end: true)
                aggressiveMemoryRelease()
            }
            self.push2TalkButton.setTitle("长按发送语音", for: .normal)
            self.push2TalkButton.backgroundColor = UIColor.aly_fromHex(0x49419E)
            print("push2TalkButtonLongPressed action")
        case .cancelled:
            print("long press cancelled")
        case .failed:
            print("long press failed")
        case .recognized:
            break
        @unknown default:
            break
        }
    }

    func aggressiveMemoryRelease() {
        _audioData.removeAll(keepingCapacity: false)
        _audioData = Data()
        _audioData.reserveCapacity(0) // 强制释放底层缓存
    }
    
    func convertToInt16LittleEndian() -> [Int16] {
        return audioData.withUnsafeBytes { rawBufferPtr -> [Int16] in
            guard rawBufferPtr.count % MemoryLayout<Int16>.stride == 0 else {
                return []
            }
            let int16Buffer = rawBufferPtr.bindMemory(to: Int16.self)
            return int16Buffer.map { Int16(littleEndian: $0) }
        }
    }
    
    @objc func logButtonTapped(sender: UIButton) {
//        let isStarted = sender.isSelected // 当前状态
//        if isStarted { // 日志已开启
//            self.videoChat.debugExportLogs(self) {
//                DispatchQueue.main.async {
//                    ALYToast.shared.showToast(text: "日志已导出", style: .success)
//                }
//            }
//        } else { // 日志未开启
//            self.videoChat.debugStartLogs()
//            self.frameIdLabel.isHidden = false
//            ALYToast.shared.showToast(text: "日志已开启", style: .success)
//        }
//        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
//            sender.isSelected = !isStarted
//        }
//    }
//
//    @objc func logButtonDoubleTapped() {

        self.videoChat.debugExportLocalAvatarAudios(self) {
            DispatchQueue.main.async {
                ALYToast.shared.showToast(text: "音频数据已导出", style: .success)
            }
        }
    }

    @objc func toggleRTButtonTapped(sender: UIButton) {
        sender.isSelected = !sender.isSelected
        let shouldShow = sender.isSelected
        self.performanceView.isHidden = !shouldShow
    }

    @objc func testAudioDataButtonTapped(sender: UIButton) {
           guard let state = self.chatState, state != .responding else {
               return
           }
           aggressiveMemoryRelease()
           guard let url = Bundle(for: Self.self).url(forResource: "Audio_16K.pcm", withExtension: nil) else {
               return
           }
           
           let sampleRate = self.dialogConfig.outboundSampleRate
           let bitDepth = 16
           
           // 将耗时操作移至后台线程
           DispatchQueue.global(qos: .userInitiated).async { [weak self] in
               guard let self = self else { return }
               
               do {
                   // 1. 读取音频数据（后台线程）
                   let audioData = try Data(contentsOf: url)
                   
                   // 2. 数据转换（后台线程）
                   let pcmSamples = self.convertToInt16LittleEndian(data: audioData)
                   print("pcmSamples \(pcmSamples)")
                   
                   // 3. 计算信息（后台线程）
                   let bytesPerSample = bitDepth / 8
                   let totalSamples = audioData.count / bytesPerSample
                   let duration = Double(totalSamples) / Double(sampleRate)
                   self._audioData = audioData
                   let success = self.videoChat.pushAudioData(pcmSamples, end: true)
                   if success ?? false {
                       print("Audio data pushed successfully")
                   }
                   self.aggressiveMemoryRelease()
                   
               } catch {
                  
               }
           }
       }

       // 改进的数据转换方法
       private func convertToInt16LittleEndian(data: Data) -> [Int16] {
           return data.withUnsafeBytes { rawBufferPointer -> [Int16] in
               let bufferPtr = rawBufferPointer.bindMemory(to: Int16.self)
               return Array(bufferPtr)
           }
       }

    @objc func muteButtonTapped(sender: UIButton) {
        sender.isSelected = !sender.isSelected
        let isMuted = sender.isSelected
        self.videoChat.muteLocalMic(mute: isMuted)
        if isMuted {
            ALYToast.shared.showToast(text: "麦克风已静音", style: .info)
        }
    }

    @objc func subtitleButtonTapped(sender: UIButton) {
        sender.isSelected = !sender.isSelected
        self.subtitleTableView.isHidden = sender.isSelected
        self.fpsLabel.isHidden = sender.isSelected
        self.frameIdLabel.isHidden = sender.isSelected
    }
}


extension VideoChatViewController: UITableViewDataSource {

    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return subtitles.count
    }

    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "subtitle") as! SubtitleTableViewCell

        let subtitle = subtitles[indexPath.row]
        cell.subtitleLabel.text = subtitle.text
        if subtitle.type == .responding {
            cell.iconView.image = UIImage(named: "tongyi_logo")
            cell.iconView.isHidden = false
            cell.updateLayout(left: 12, right: 48, iconWidth: 24)
        } else {
            cell.iconView.image = nil
            cell.iconView.isHidden = true
            cell.updateLayout(left: 48, right: 12, iconWidth: 0)
        }
        return cell
    }
}

class SubtitleTableViewCell: UITableViewCell {

    let iconView = UIImageView()
    let subtitleLabel = UILabel()
    let mainView = UIView()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)

        createView()
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    func createView() {
        self.backgroundColor = .clear

        mainView.backgroundColor = .white
        mainView.clipsToBounds = true
        mainView.layer.cornerRadius = 6

        contentView.addSubview(mainView)
        mainView.snp.makeConstraints { make in
            make.leading.equalToSuperview().inset(8)
            make.top.bottom.trailing.equalToSuperview().inset(4)
        }
        mainView.addSubview(iconView)
        mainView.addSubview(subtitleLabel)
        iconView.snp.makeConstraints { make in
            make.leading.equalToSuperview().inset(8)
            make.top.equalToSuperview().inset(8)
            make.height.equalTo(24)
            make.width.equalTo(24)
        }
        subtitleLabel.snp.makeConstraints { make in
            make.leading.equalTo(iconView.snp.trailing).offset(12)
            make.top.bottom.trailing.equalToSuperview().inset(4)
        }
        subtitleLabel.numberOfLines = 0
        subtitleLabel.textAlignment = .left
        subtitleLabel.textColor = .black
        subtitleLabel.font = .systemFont(ofSize: 14)
        iconView.contentMode = .scaleAspectFit
    }

    func updateLayout(left: Int, right: Int, iconWidth: Int) {
        mainView.snp.updateConstraints { make in
            make.leading.equalToSuperview().inset(left)
            make.trailing.equalToSuperview().inset(right)
        }
        iconView.snp.updateConstraints { make in
            make.width.equalTo(iconWidth)
        }
    }
}

extension VideoChatViewController : TYAudioRecorderDelegate {
    func recorderDidStart() {
      
    }
    
    func recorderDidStop() {
        
    }
    
    func voiceRecorded(_ buffer: UnsafeMutablePointer<UInt8>, length: Int32) {
        print("[VideoChatViewController] voiceRecorded")
        let byteCount = Int(length)
        // 使用安全的内存访问模式
        let chunk = Data(bytes: buffer, count: byteCount)
        // 原子化追加操作
        audioDataQueue.async(flags: .barrier) {
            self._audioData.append(chunk)
            // 内存优化：超过阈值时压缩存储
            if self._audioData.count > 1024 * 1024 * 30 { // 10MB阈值
                self._audioData.removeAll(keepingCapacity: true)
                self._audioData.reserveCapacity(1024 * 1024 * 5) // 保留5MB空间
            }
        }
    }
    
    func recorderDidFail(_ error: (any Error)?) {
        
    }
}
